% driver: parasitic_currents.m
% make a nice example image of this phenomenon
% dnelson dec2012

clear all; clc;

global nx ny dt nstep Nf;
nx = 50;
ny = 50;
Nf = 200;
dt = 0.00125/2;
nstep = 250;

% config
setup_globals_area_cons;

% set initial density and viscosity in the box and in the drop
setup_ICs;

% setup front
[xf,yf,xf_start,yf_start] = setup_front_drop(xc,yc,rad,Nf);

% timestep loop
for curstep=1:nstep

    fprintf('[%3d] time = %7.5f \n',curstep,time);

    % save solution state at beginning of Delta_x step
    timestep_helper(1);

    % take two substeps for the second order time integration
    for substep=1:2

        % calculate surface tension (and map onto grid)
        calc_surface_tension(xf,yf,Nf,1);

        % calculate u_dagger neglecting pressure
        calc_vel_adv_diff;

        % apply pressure correction to u_dagger to obtain u^(n+1)
        pressure_correction_vel;

        % advect the front using corrected velocity field
        [xf,yf,u1f,u2f] = front_advect(xf,yf,Nf);

        % reconstruct density and viscosity everywhere on the domain
        reconstruct_density_mu(xf,yf,Nf,rho2,mu2,1);

    end

    % update the solution vector using the predictor-corrector average
    timestep_helper(2);

    time = time+dt;

    % maintain good spacing of the front points (by adding/deleting)
    [xf,yf,Nf] = front_maintaince(xf,yf,Nf);
end

% interpolate velocity mag to smooth
ni = 200;
[xx,yy] = meshgrid(x(2:nx+1),y(2:ny+1));
vvv = sqrt(u1(1:nx,2:nx+1).^2 + u2(2:ny+1,1:ny).^2);
[xx_i,yy_i] = meshgrid(linspace(0,1,ni),linspace(0,1,ni));
vvv_i = interp2(xx,yy,vvv,xx_i,yy_i,'spline');

% plot filled contour of velocity magnitude
colormap(jet)
colormap(flipud(colormap))
contourf(xx_i,yy_i,vvv_i,10);

hold on
axis equal
xlim([0 Lx])
ylim([0 Ly])
plot([xf(1:Nf) xf(1)],[yf(1:Nf) yf(1)],'k','linewidth',2); % drop interface
colorbar
