% driver: area_conservation.m
% how well is the drop area conserved in a static scenario
% dnelson dec2012

clear all; clc;

% outer iteration over fixed grid resolution
global nx ny dt nstep Nf;
nn_vals = [16,32,64];
Nf_vals = [50,100,200];
dt_vals = [0.00125*2,0.00125,0.00125/2];
nstep_vals = [50,100,200];
colors = ['r' 'g','b'];

for res_iter=1:length(nn_vals)
    nx = nn_vals(res_iter);
    ny = nn_vals(res_iter);
    Nf = Nf_vals(res_iter);
    dt = dt_vals(res_iter);
    nstep = nstep_vals(res_iter);
    
    % config
    setup_globals_area_cons;    

    % set initial density and viscosity in the box and in the drop
    setup_ICs; 

    drop_area = zeros(nstep,1);

    % setup front
    [xf,yf,xf_start,yf_start] = setup_front_drop(xc,yc,rad,Nf);

	% timestep loop
	for curstep=1:nstep

        fprintf('[%3d] time = %7.5f \n',curstep,time);

        % save solution state at beginning of Delta_x step
        timestep_helper(1);

        % take two substeps for the second order time integration
        for substep=1:2

            % calculate surface tension (and map onto grid)
            calc_surface_tension(xf,yf,Nf,1);

            % calculate u_dagger neglecting pressure
            calc_vel_adv_diff;

            % apply pressure correction to u_dagger to obtain u^(n+1)
            pressure_correction_vel;

            % advect the front using corrected velocity field
            [xf,yf,u1f,u2f] = front_advect(xf,yf,Nf);

            % reconstruct density and viscosity everywhere on the domain
            reconstruct_density_mu(xf,yf,Nf,rho2,mu2,1);

        end

        % update the solution vector using the predictor-corrector average
        timestep_helper(2);

        time = time+dt;

        % maintain good spacing of the front points (by adding/deleting)
        [xf,yf,Nf] = front_maintaince(xf,yf,Nf);

        % calculate drop area and save
        drop_area(curstep) = 0.5 * sum( xf(1:Nf+1).*yf(2:Nf+2) - ...
                                        xf(2:Nf+2).*yf(1:Nf+1) );
	end

    % plot area vs. time
    plot(linspace(0,dt*nstep,nstep),drop_area./drop_area(1),colors(res_iter),'linewidth',2);
    hold on
end % outer iteration

% finish plot
xlim([0 dt*nstep]);
xlabel('time')
ylabel('drop area / initial drop area')
legend('nx = ny = 16','nx = ny = 32','nx = ny = 48')
